############################################################################
# study147.py
# Script to run Study 147 (Learning Optimization, Experiment 2 -
# IFB vs. DFB with MC/CR)
#
# Author: Troy Smith
# Last updated 10/20/2008 09:15
#
############################################################################

import os, sys, numpy, datetime, time
from stimulus import Stimulus
from splash import StartWindow
from demographics import DemographicWindow
from Tkinter import *
import tkMessageBox

# Define a custom class for the main display window
class DisplayWindow(Toplevel):

    def __init__(self, parent=None, nChoices=4):
        # do superclass init
        Toplevel.__init__(self, parent)
        # make the window cover the full screen
        w, h = self.winfo_screenwidth(), self.winfo_screenheight()
        self.overrideredirect(1)
        self.geometry('%dx%d+0+0' % (w, h))
        # initialize instance attributes
        self.nChoices = nChoices
        self.choices = []
        self.crRespVar = StringVar()
        self.mcRespVar = StringVar()
        # set up the interface widgets
        # Empty label for top 100 pixels
        Label(self, text='', pady=50).pack(side=TOP)
        # Stimulus
        self.stimLabel = Label(self, text='', pady=20,
            justify=LEFT, font=('times', 28, 'bold'), 
            wraplength='960', fg='blue')
        self.stimLabel.pack(side=TOP, anchor=W, padx=10)
        # Response
        self.respLabel = Label(self, text='', pady=20,
            justify=LEFT, font=('times', 28, 'bold'), fg='red')
        self.respLabel.pack(side=TOP, anchor=W, padx=10)
        # Subject response input for text entry (constructed response)
        self.crResp = Entry(self, width=50, textvariable=self.crRespVar,
            font=('times', 28, 'bold'))
        self.crResp.pack(side=TOP, anchor=NW, pady=5, padx=10)
        # Empty label for next 30 pixels
        Label(self, text='', pady=15).pack(side=TOP)
        # Radio buttons for multiple choice
        for n in range(self.nChoices):
            self.choices.append(Radiobutton(self, \
                text='', variable=self.mcRespVar, value='',
                font=('times', 28, 'bold')))
            self.choices[n].pack(side=TOP, anchor=NW, pady=5, padx=10)

    def erase(self):
        self.stimLabel.config(text='')
        self.respLabel.config(text='')
        self.crRespVar.set('')
        self.mcRespVar.set('')
        self.crResp.config(state=DISABLED)
        for button in self.choices:
            button.config(text='', value='', state=DISABLED)

    def hideMC(self):
        for button in self.choices:
            button.config(state='disabled', text='', value='')
    
    def showMC(self, choices=[], randomize=True):
        nChoices = len(choices)
        # Verify that the number of distractors is within the capacity
        # of the radio button array
        if nChoices > self.nChoices:
            errMsg = "Fatal error in stimdisplay.showMC ...\n"
            errMsg += "Too many choices input...\n"
            errMsg += "Input %d, but only %d are allowed" % \
            	(nChoices, self.nChoices)
            tkMessageBox.showerror('ERROR', errMsg)
            root.quit()
            sys.exit(errMsg)
        # Randomize the choices before displaying them
        if randomize: numpy.random.shuffle(choices)
        # Hide all the buttons
        self.hideMC
        # Assign the choices to the buttons and enable the buttons
        for n in range(nChoices):
            self.choices[n].config(\
                text=choices[n], value=choices[n], state=NORMAL)
        
if __name__ == '__main__':
    # Initialize variables
    # Experiment info
    exptName = 'Study 147 - SAT/GRE vocabulary'
    exptShort = 'S147v1'
    nSessions = 2
    trialTime = 12000
    countDownTime = 15
    nMCchoices = 4
    # Date and time
    d = datetime.datetime.now()
    today = d.date().isoformat()            # YYYY-MM-DD
    startTime = d.time().isoformat()[:8]    # HH:MM:SS

    # Create a root Tkinter window and hide the root window
    root = Tk()
    root.withdraw()

    # Display splash screen and get participant and session info
    start = StartWindow(root, exptName, nSessions)
    pid = start.pid
    curSession = start.curSession
    location = start.location
    
    # Input and output file info
    stimfile = '%s%sstimuli%sstimuli_%03d_%d.txt' % \
        (os.curdir, os.sep, os.sep, pid, curSession)
    dp = '%s%sdata%s' % (os.curdir, os.sep, os.sep)
    outfile_main = '%s%s.data%s.txt' % (dp, exptShort, location)
    outfile_dated = '%s%s.data.%s.%s.txt' % (dp, exptShort, today, location)
    outfile_subj = '%s%s.data.%03d_%d.%s.txt' % \
        (dp, exptShort, pid, curSession, location)

    # Read in stimuli from text file
    stimuli = []
    nStim = 0
    try:
        fin = open(stimfile, 'rU')
        while True:
            text = fin.readline()
            # Stop reading when EOF is encountered
            if text == "":
                break
            # Skip any commented lines
            elif text[0] == '#':
                continue
            # Parse line and create a new Stimuli object
            else:
                text = text.replace('\n','')
                parsed = text.split('\t')
                #print parsed
                # Fields should be:
                #   Seq, CondCode, CondName, PresNum, PresType,
                #   StimID, Stim, Resp, Dist1, Dist2, Dist3
                if len(parsed) < 8:
                    # Throw exception. Not enough fields.
                    raise IOError
                else:
                    # create a new Stimuli object
                    stimuli.append(Stimulus())
                    # Set Stimulus object attributes
                    stimuli[nStim].session  = curSession
                    stimuli[nStim].seq      = int(parsed[0])
                    stimuli[nStim].condCode = int(parsed[1])
                    stimuli[nStim].condName = parsed[2]
                    stimuli[nStim].presNum  = int(parsed[3])
                    stimuli[nStim].presType = parsed[4]
                    stimuli[nStim].stimID   = int(parsed[5])
                    stimuli[nStim].stim     = parsed[6]
                    stimuli[nStim].corrResp = parsed[7]
                    if stimuli[nStim].presType == 'MC':
                        stimuli[nStim].distractors = parsed[8:11]
                    else:
                        stimuli[nStim].distractors = []
                    nStim += 1
    except IOError:
        if os.path.exists(stimfile):
            errMsg = "Error reading stimulus file '%s'\n" % (stimfile)
        else:
            errMsg = "Fatal error...\n"
            errMsg += "Stimulus file '%s' not found\n" % (stimfile)
        tkMessageBox.showerror('ERROR', errMsg)
        root.quit()
        sys.exit(errMsg)
    else:
        fin.close()

    # Before starting the first session, get demographic data and
    # run through the demo
    if curSession == 1:
        # Get demographics info and save to file
        dmg = DemographicWindow(root, exptShort, pid)
        def submitdemographics(event):
            dmg.submitform()
            dmg.quit()
        dmg.ok.bind('<Return>', submitdemographics)
        dmg.ok.bind('<Button-1>', submitdemographics)
        dmg.lift()
        dmg.mainloop()
        
        # Create the demo window to display stimuli and get participant
        # responses
        demoWin = DisplayWindow(root)
        
        index = 0
        demoStim = []
        s = 'tending to improve; beneficial; wholesome'
        r =	'salutary'
        d = ['vehement', 'dynamo',	'nubile']
        demoStim.append((s,r,d))
        s = 'blessedness; state of great happiness'
        r =	'beatitude'
        d =	['fortuitous', 'regiment', 'conscientious']
        demoStim.append((s,r,d))
        def demoloop(event):
            global demoStim, index
            demoWin.erase()
            index += 1
            if index == 1:
                stim = demoStim[0][0]
                resp = demoStim[0][1]
                demoWin.stimLabel.config(text=stim)
                demoWin.respLabel.config(text=resp)
                demoWin.crResp.config(state=NORMAL)
            elif index == 2:
                stim = demoStim[1][0]
                resp = demoStim[1][1]
                demoWin.stimLabel.config(text=stim)
                demoWin.respLabel.config(text=resp)
                demoWin.crResp.config(state=NORMAL)
            elif index == 3:
                stim = demoStim[0][0]
                resp = ''
                demoWin.stimLabel.config(text=stim)
                demoWin.respLabel.config(text=resp)
                demoWin.crResp.config(state=NORMAL)
            elif index == 4:
                stim = demoStim[0][0]
                resp = ''
                chc = ['beatitude', 'fortuitous', 'regiment', 'conscientious']
                demoWin.stimLabel.config(text=stim)
                demoWin.respLabel.config(text=resp)
                demoWin.showMC(chc)
            elif index == 5:
                stim = demoStim[1][0]
                resp = demoStim[1][1]
                demoWin.stimLabel.config(text=stim)
                demoWin.respLabel.config(text=resp)               
                demoWin.crResp.config(state=NORMAL)
            elif index == 6:
                stim = demoStim[0][0]
                resp = demoStim[0][1]
                demoWin.stimLabel.config(text=stim)
                demoWin.respLabel.config(text=resp)                
                demoWin.crResp.config(state=NORMAL)
            else:
                demoWin.quit()
                return
                    
        # Bind the F1 key to advance the display
        demoWin.bind('<F1>', demoloop)
        demoWin.lift()
        demoWin.mainloop()

    # Show a warning box before beginning the main experiment
    t = 'Press OK to begin session %s' % (curSession)
    tkMessageBox.showinfo('Ready to start',t)
    
    # Create the main window to display stimuli and get participant responses
    mainWin = DisplayWindow(root, nChoices=nMCchoices)

    # Define a time delayed loop through the stimuli
    index = -1
    lastcall = []
    def stimloop():
        global stimuli, index, lastcall, curSession, nSessions, nStim
        global outfile_main, outfile_dated, outfile_subj, trialTime
        # Store performance data from last trial
        if lastcall:
            # Get data
            if stimuli[index].presType == 'MC':
                stimuli[index].subjResp = mainWin.mcRespVar.get()
            else:
                stimuli[index].subjResp = mainWin.crRespVar.get()
            # Score trial
            stimuli[index].scoreTrial()
            # Write data to output files
            outData = ''
            if index == 0:
                # Write header for first stimulus of this session
                outData += stimuli[index].__str__('header')
            outData += stimuli[index].__str__()
            f = open(outfile_main, 'a')
            f.write(outData)
            f.close()
            f = open(outfile_dated, 'a')
            f.write(outData)
            f.close()
            f = open(outfile_subj, 'a')
            f.write(outData)
            f.close()
            
        # If there are no more trials, kill window and exit
        if index >= nStim -1:
            # Cancel the last call to stimloop
            root.after_cancel(lastcall)
            # Close main window and display final screen
            mainWin.destroy()
            t = "Thank you for participating.\n"
            if curSession == nSessions:
                # Done with experiment
                t += "Please inform the experimenter that you are "
                t += "done and ready for a short debriefing.\n"
            else:
                # Remind participant of remaining sessions
                t += "Please inform the experimenter that you are done "
                t += "with this session.\n\n"
            tkMessageBox.showinfo('Finished with session', t)
            # End the program
            root.quit()
        # Otherwise, continue with the next trial
        else:    
            index += 1
            # Erase previous display
            mainWin.erase()
            # Show stimulus
            mainWin.stimLabel.config(text=stimuli[index].stim)
            # On cued recall test trials, hide answer and show response box
            if stimuli[index].presType == 'CR':
                mainWin.respLabel.config(text='')
                mainWin.crResp.config(state=NORMAL)
                mainWin.hideMC()
            # On multiple choice test trials, hide answer and show choices
            elif stimuli[index].presType == 'MC':
                mainWin.respLabel.config(text='')
                mainWin.crResp.config(state=DISABLED)
                choices = [stimuli[index].corrResp] + stimuli[index].distractors
                mainWin.showMC(choices)
            # On study or feedback trials, show answer and response box
            else:
                mainWin.respLabel.config(text=stimuli[index].corrResp)
                mainWin.crResp.config(state=NORMAL)
                mainWin.hideMC()
            # Place a delayed call to stimloop for the end of the trial
            lastcall = root.after(trialTime, stimloop)

    # Define a function to wait 15 seconds before starting the
    # stimulus presentation loop
    countdown, counter, startstim = [], countDownTime, []
    def waitloop():
        global countdown, counter, startstim
        if counter > 1:
            # continue counting down
            t = "The session will begin in %d seconds" % (counter)
            mainWin.stimLabel.config(text=t)
            counter -= 1
            countdown = root.after(1000, waitloop)
        else:
            # start the stimulus presentation loop
            startstim = root.after(1000, stimloop)

    # Start the countdown
    mainWin.erase()
    waitloop()
    # Start the main loop
    mainWin.mainloop()
    # Close program when done
    root.quit()
